/******************************************************************************

	Cholesky decomposition of a symmetric positive-definite matrix
	A = L * L^T


	input:
	  APtr,AKey,AMat	matrix to be decomposed
	  N			dimension of matrix A 
	  thresh		sparse threshold
				threshold below 10^-6 is not recommended
	  semithresh		criterium for checking positive definitness
				threshold below 10^-10 is not recommended
	  semidefinite		false = ok, true  = definitness endangered
	  ok			true  = ok, false = negative radicant

	output:
	  LPtr,LKey,LMat	Cholesky factor (lower triangular)
	  LTPtr,LTKey,LTMat	Cholesky factor (upper triangular)

******************************************************************************/


#include "sparsecholes.h"

int sp_chol_decomp_csr(int *&LPtr, int *&LKey, double *&LMat, int *LTPtr, 
      int *LTKey, double *LTMat, int *APtr, int *AKey, double *AMat, int *N, 
      double *thresh, double *semithresh, bool *semidefinite, bool *ok){

//=============================================================================
// remove lower triangular of input matrix

  sp_killtri_csr(APtr, AKey, AMat, N);

//=============================================================================
// decomposition 

  //allocation 
  int     i,j,k,l;
  int     N2   = (*N)*(*N);
  int     half = ((*N)+1)*(*N)/2;
  int     *KeySwap = NULL;

  int    *LhelpPtr    = new int[(*N)+1];
  int    *LhelpKey    = new int[half];
  double *LhelpMat    = new double[half];
                      
  int    *RowEndPtr   = new int[(*N)];
  int    *OffsetsLT   = new int[(*N)];
  int    *Index       = new int[(*N)];
  int    *TmpKey      = new int[(*N)+1];
  int    *MergedKeys  = new int[(*N)+1]; 

  double *CurrentColL = new double[(*N)];
  double *SemiDef     = new double[(*N)];

  int    ltidx,aj,aj1,arowlen,lj,lj1,lrowlen,keylen,jx,
         lk,lk1,lt,ltk,ltk1,ltdim,m,el_act;
  double Ljk,diag,diag_inv,Lij;

  for(i = 0; i < (*N); i++)   LhelpPtr[i]    = 0;
  for(i = 0; i < (*N); i++)   RowEndPtr[i]   = 0;
  for(i = 0; i < (*N); i++)   OffsetsLT[i]   = 0;
  for(i = 0; i < (*N); i++)   CurrentColL[i] = 0;
  for(i = 0; i < (*N); i++)   SemiDef[i]     = 0;
  for(i = 0; i < half; i++)   LhelpKey[i]    = 0;
  for(i = 0; i < half; i++)   LhelpMat[i]    = 0.e0;
  for(i = 0; i < (*N)+1; i++) TmpKey[i]      = 0;
  for(i = 0; i < (*N)+1; i++) MergedKeys[i]  = 0;

  //first settings
  for(i = 1; i < (*N)+1; i++){
    LhelpPtr[i] = LhelpPtr[i-1] + i;
  }
  LTPtr[0] = 0;
  ltidx    = 0;
  jx       = 0;

  //loop over columns 
  for (j = 0; j < (*N); j++){

    //get offsets for forming diagonal elements
    aj      = APtr[j];		//begin of current row of A
    aj1     = APtr[j+1];	//begin of next row of A
    arowlen = aj1 - aj;

    lj      = LhelpPtr[j]; 	//begin of current row in L
    lrowlen = RowEndPtr[j];  	//length of current row in L
    lj1     = lj + lrowlen; 	//index for end of current row in L

    diag    = AMat[aj];

    //form diagonal elements
    for(lj; lj < lj1; lj++){
      Ljk = LhelpMat[lj];
      diag -= Ljk*Ljk;		//a_jj-l_jk^2
    }

    //check if radicant is ok, otherwise return
    if (diag <= 0.e0){
      *ok = false;
      return ERR_NEGATIVE_RADICANT;
    }

    //check if positive definitness is endangered
    if(diag <= *semithresh){
      SemiDef[j] = *semithresh - diag;
      diag       = *semithresh;
      *semidefinite = true;
    }

    diag     = sqrt(diag);	
    diag_inv = 1.e0/diag;

    //write ptr offset for LT
    LTPtr[j] = ltidx;
    
    //write keys of current row of A in MergedKeys
    KeySwap = AKey + aj;
    for(i = 0; i < arowlen; i++){ 
      MergedKeys[i] = KeySwap[i];	
    }
    MergedKeys[arowlen] = arowlen;
    KeySwap = NULL;
    keylen  = arowlen;

    //scale elements in row j of A with diag L_jj and put in CurrentColL
    jx++;
    for (aj; aj < aj1; aj++){
      i = AKey[aj];
      CurrentColL[i] = AMat[aj] * diag_inv;	
      Index[i]       = jx;		//save information where A_ji are placed in CurrentColL
    }

    //form l_ki^T*l_jk with k=1,...,j-1
    lj = LhelpPtr[j];
    for(lj; lj < lj1; lj++){
      //scale element l_jk in row j of LhelpMat with diag LT_jj
      Ljk  = LhelpMat[lj];
      Ljk *= diag_inv;
      lk   = LhelpKey[lj];		//kth col of L = kth row of LT
      ltk1 = LTPtr[lk+1];
      //find ltk = LT_ki 
      if (OffsetsLT[lk] != 0){			
        ltk = OffsetsLT[lk];
        if (LTKey[ltk] <= j && ltk < ltk1){	//start search at offset known from previous cycle
          ltk++;
        }
        OffsetsLT[lk] = ltk;			//store offset in LT for later access
      }else{					//search, if no offset is known 
        ltk = LTPtr[lk];		
        while (LTKey[ltk] <= j && ltk < ltk1){
          ltk++;
        }
        OffsetsLT[lk] = ltk;			//store offset in LT for later access
      }
      ltdim = ltk1 - ltk;

      //merge keys of current rows of A and L^T in increasing order
      sp_keymerge_csr(MergedKeys, LTKey+ltk, TmpKey, &keylen, &ltdim, N);

      // swap pointers
      KeySwap    = MergedKeys;
      MergedKeys = TmpKey;
      TmpKey     = KeySwap;
      KeySwap    = NULL;

      //multiply l_jk with kth row of LT
      for (ltk; ltk < ltk1; ltk++){
        i = LTKey[ltk];
        //if no fill in element, subtract from a_ji/l_jj
        if (Index[i] == jx){			
          CurrentColL[i] -= LTMat[ltk] * Ljk;
        //if fill in element, set value
        }else{				
          CurrentColL[i] = -LTMat[ltk] * Ljk;
          Index[i]      = jx;
        }
      }
    }

    //store diags in LT
    LTMat[ltidx] = diag;
    LTKey[ltidx] = j;
    ltidx++;

    //search starting point for screening CurrentColL
    m = 0;
    while(m < keylen){
      i = MergedKeys[m];
      if (i > j){ 		//lower triangular part of CurrentColL is needed
        break;
      }
      m++; 
    }

    //scatter current col on Lhelp and LT
    for (m; m < keylen; m++){
      i = MergedKeys[m];	//row of L
      Lij = CurrentColL[i]; 
      if (fabs(Lij) > *thresh){
        el_act           = LhelpPtr[i] + RowEndPtr[i];
        LhelpMat[el_act] = Lij;
        LhelpKey[el_act] = j;
        LTMat[ltidx]     = Lij;
        LTKey[ltidx]     = i;
        RowEndPtr[i]++;
        ltidx++;
      }
    }
  
  }

  LTPtr[(*N)] = ltidx;

  delete[] LhelpPtr;
  delete[] LhelpKey;
  delete[] LhelpMat; 

  delete[] RowEndPtr;
  delete[] OffsetsLT; 
  delete[] Index;  
  delete[] TmpKey;    
  delete[] MergedKeys; 

  delete[] CurrentColL; 
  delete[] SemiDef;     

  delete[] APtr;
  delete[] AKey;
  delete[] AMat;

//=============================================================================

  if(*semidefinite == true){
    printf("Warning: Matrix seems to be semidefinite!\n");
    printf("Maybe you have to decrease semithresh. It is currently set to %e. \n",semithresh);
    printf("It is recommened to set semithresh not lower than 10^(-10).\n");
  }

//=============================================================================
// transpose

  LPtr  = new int[(*N)+1];
  LKey  = new int[(*N)*(*N)];
  LMat  = new double[(*N)*(*N)];
  
  for(i = 0; i < (*N)+1; i++)     LPtr[i]  = 0;
  for(i = 0; i < (*N)*(*N); i++)  LKey[i]  = 0;
  for(i = 0; i < (*N)*(*N); i++)  LMat[i]  = 0.e0;

  sp_trapos_csr(LTPtr,LTKey,LTMat,LPtr,LKey,LMat,N);

//=============================================================================

}
