#!/usr/bin/env python

import sys
import os
import re
from optparse import OptionParser
import numpy
import numpy.fft
import math
from scipy import integrate

# Constants and Variables

kb   = 3.166815343e-6   		# Boltzmann in a.u./K
c    = 2.99792458e+10   		# Speed of light in cm/s
h    = 6.626070040e-34/4.35974417e-18   # Planck constant in a.u. * s

# The Parser 

def parse(opts, lines):

  the_data = []
  for l in lines:
    if not l.startswith("#"):
      try:
        vals = l.strip().split()
      except:
        raise Exception ("Failed to parse line...")

      try:
        wvnin = float(vals[0])
        intin = float(vals[1]) 
      except:
        raise Exception ("Failed to convert data... ")

      the_data.append ([wvnin, intin]) #x,y,z

  ndata = len(the_data)

  if ndata <= 0:
    raise Exception ("Failed to find any data, check target and options.")

  return the_data

def intrange(opts, npts, dnu):

  nu_max = npts
  nu_min = 1

  if opts.nu_max != 0:
    nu_max = int(round(opts.nu_max*c/dnu))

  if opts.nu_min != 0:
    nu_min = int(round(opts.nu_min*c/dnu))

  return (nu_min,nu_max)

def freedom(the_data, opts, beta):

  deg_free = 0.e0
  np_data = numpy.einsum('ij->ji',numpy.asarray(the_data))
  npts = np_data.shape[1]

  nu = np_data[0,:] * c # nu in 1/s
  dnu = (nu[1]-nu[0])   # in 1/s

  (nu_min,nu_max) = intrange(opts,npts,dnu)

  array = [0 for x in range(nu_min-1,nu_max)]

  i = 0
  if opts.do_blc=="true":
    for l in range(nu_min-1,nu_max):
      array[i] = np_data[1,l] - np_data[1,nu_max-1] # Set last value to zero
      i = i + 1
  else:
    for l in range(nu_min-1,nu_max):
      array[i] = np_data[1,l] # Set last value to zero
      i = i + 1

  if opts.int_method == "trapz":
    deg_free = numpy.trapz(array,dx = dnu)
  elif opts.int_method == "simps":
    deg_free = integrate.simps(array,dx = dnu)
  else:
    die("Illegal Integration Method!")

  return (deg_free * beta)

def corr_energy(the_data, opts, beta):

  energy_cl = 0.e0
  energy_qm = 0.e0
  np_data = numpy.einsum('ij->ji',numpy.asarray(the_data))
  npts = np_data.shape[1]

  nu   = np_data[0,:] * c # nu in 1/s

  dnu  = (nu[1]-nu[0])    #in 1/s
  bhnu = nu * h * beta    #no units

  (nu_min,nu_max) = intrange(opts,npts,dnu)

  #------------------- Classic -------------------

  array_cl = [0 for x in range(nu_min-1,nu_max)]
  i = 0

  for l in range(nu_min-1,nu_max):
    array_cl[i] = np_data[1,l]
    i = i + 1

  if opts.int_method == "trapz":
    energy_cl = numpy.trapz(array_cl,dx = dnu)
  elif opts.int_method == "simps":
    energy_cl = integrate.simps(array_cl,dx = dnu)
  else:
    die("Illegal Integration Method!")

  #------------------- Quantum -------------------

  array_qm = [0 for x in range(nu_min-1,nu_max)]
  i = 0

  for l in range(nu_min-1,nu_max):
    if bhnu[l] == 0:
      array_qm[i] = np_data[1,l] * (0.5e0 * bhnu[l] + 1.e0)
      i = i + 1
    else:
      array_qm[i] = np_data[1,l] * (0.5e0 * bhnu[l] + bhnu[l]/(numpy.exp(bhnu[l])-1.e0))
      i = i + 1

  if opts.int_method == "trapz":
    energy_qm = numpy.trapz(array_qm,dx = dnu)
  elif opts.int_method == "simps":
    energy_qm = integrate.simps(array_qm,dx = dnu)
  else:
    die("Illegal Integration Method!")

  return (energy_cl,energy_qm)

def corr_entropy(the_data, opts, beta):

  entropy_cl = 0.e0
  entropy_qm = 0.e0
  np_data = numpy.einsum('ij->ji',numpy.asarray(the_data))
  npts = np_data.shape[1]

  nu   = np_data[0,:] * c # nu in 1/s

  dnu  = (nu[1]-nu[0])   #in 1/s
  bhnu = nu * h * beta   #no units

  (nu_min,nu_max) = intrange(opts,npts,dnu)

  #------------------- Classic -------------------

  array_cl = [0 for x in range(nu_min-1,nu_max)]
  i = 0

  for l in range(nu_min-1,nu_max):
    if bhnu[l] == 0:
      array_cl[i] = 0.e0#np_data[1,l+1] * (1.e0 - numpy.log(bhnu[l+1]))
      i = i + 1
    else:
      array_cl[i] = np_data[1,l] * (1.e0 - numpy.log(bhnu[l]))
      i = i + 1

  if opts.int_method == "trapz":
    entropy_cl = numpy.trapz(array_cl,dx = dnu)
  elif opts.int_method == "simps":
    entropy_cl = integrate.simps(array_cl,dx = dnu)
  else:
    die("Illegal Integration Method!")

  #------------------- Quantum -------------------

  array_qm = [0 for x in range(nu_min-1,nu_max)]
  i = 0

  for l in range(nu_min-1,nu_max):
    if bhnu[l] == 0:
      array_qm[i] = 0.e0# np_data[1,l+1] * (bhnu[l+1]/(numpy.exp(bhnu[l+1])-1.e0) - numpy.log(1.e0 - numpy.exp(-bhnu[l+1])))
      i = i + 1
    else:
      array_qm[i] = np_data[1,l] * (bhnu[l]/(numpy.exp(bhnu[l])-1.e0) - numpy.log(1.e0 - numpy.exp(-bhnu[l])))
      i = i + 1

  if opts.int_method == "trapz":
    entropy_qm = numpy.trapz(array_qm,dx = dnu)
  elif opts.int_method == "simps":
    entropy_qm = integrate.simps(array_qm,dx = dnu)
  else:
    die("Illegal Integration Method!")

  return (entropy_cl/opts.temp,entropy_qm/opts.temp)

def corr_free(the_data, opts, beta):

  free_cl = 0.e0
  free_qm = 0.e0

  np_data = numpy.einsum('ij->ji',numpy.asarray(the_data))
  npts = np_data.shape[1]

  nu   = np_data[0,:] * c # nu in 1/s

  dnu  = (nu[1]-nu[0])    #in 1/s
  bhnu = nu * h * beta    #no units

  (nu_min,nu_max) = intrange(opts,npts,dnu)

  #------------------- Classic -------------------

  array_cl = [0 for x in range(nu_min-1,nu_max)]
  i = 0

  for l in range(nu_min-1,nu_max):
    if bhnu[l] == 0:
      array_cl[i] = 0.e0#np_data[1,l+1] * numpy.log(bhnu[l+1])
      i = i + 1
    else:
      array_cl[i] = np_data[1,l] * numpy.log(bhnu[l])
      i = i + 1

  if opts.int_method == "trapz":
    free_cl = numpy.trapz(array_cl,dx = dnu)
  elif opts.int_method == "simps":
    free_cl = integrate.simps(array_cl,dx = dnu)
  else:
    die("Illegal Integration Method!")

  #------------------- Quantum -------------------

  array_qm = [0 for x in range(nu_min-1,nu_max)]
  i = 0

  for l in range(nu_min-1,nu_max):
    if bhnu[l] == 0:
      array_qm[i] = 0.e0#np_data[1,l+1] * (numpy.log((1.e0 - numpy.exp(-bhnu[l+1]))/(numpy.exp(-bhnu[l+1]/2.e0))))
      i = i + 1
    else:
      array_qm[i] = np_data[1,l] * (numpy.log((1.e0 - numpy.exp(-bhnu[l]))/(numpy.exp(-bhnu[l]/2.e0))))
      i = i + 1

  if opts.int_method == "trapz":
    free_qm = numpy.trapz(array_qm,dx = dnu)
  elif opts.int_method == "simps":
    free_qm = integrate.simps(array_qm,dx = dnu)
  else:
    die("Illegal Integration Method!")

  return (free_cl,free_qm)

def combine(final_data, the_data, opts, loop, deg_free):

  if loop == 0:
    np_data = numpy.einsum('ij->ji',numpy.asarray(the_data))
    npts    = np_data.shape[1]

    final_data = []
    if loop < opts.neducts:
      sign = -1.e0
    else:
      sign =  1.e0

    if opts.nfree == 0.0:
      fac = sign
    else:
      fac = sign * opts.nfree/deg_free

    if opts.do_blc=="true":
      corr = np_data[1,npts-1]
    else:
      corr = 0.e0

    for l in range(0,npts):
      final_data.append([np_data[0,l],fac*(np_data[1,l]-corr)])

  else:
    np_data_final = numpy.einsum('ij->ji',numpy.asarray(final_data))
    np_data       = numpy.einsum('ij->ji',numpy.asarray(the_data))
    npts = np_data.shape[1]
    final_data = []
    if loop < opts.neducts:
      sign = -1.e0
    else:
      sign =  1.e0

    if opts.nfree == 0.0:
      fac = sign
    else:
      fac = sign * opts.nfree/deg_free

    if opts.do_blc=="true":
      corr = np_data[1,npts-1]
    else:
      corr = 0.e0

    for l in range(0,npts):
      final_data.append([np_data_final[0,l],np_data_final[1,l]+fac*(np_data[1,l]-corr)])

  return final_data

op = OptionParser(usage="./ThermoTraj.py power-spectrum [options]\n\n"+
                        "calculate thermodynamics from spectrum...\n")

op.add_option ("-t", "--temperature", type="float", dest="temp",
               help="# average temperature", metavar="DOUBLE")
op.add_option ("-s", "--start-wavenumber", type="float", dest="nu_min",
               help="# start wavenumber [cm^-1] for integration", metavar="DOUBLE")
op.add_option ("-m", "--max-wavenumber", type="float", dest="nu_max",
               help="# maximum wavenumber [cm^-1] for integration", metavar="DOUBLE")
op.add_option ("-n", "--ndegfree", type="float", dest="nfree",
               help="# number of degrees of freedom", metavar="DOUBLE")
op.add_option ("-e", "--neducts", type="int", dest="neducts",
               help="# number of educts", metavar="INT")
op.add_option ("-p", "--nproducts", type="int", dest="nproducts",
               help="# number of products", metavar="INT")
op.add_option ("-b", "--corr-baseline", type="string", dest="do_blc",
               help="# do baseline correction", metavar="STR")
op.add_option ("-i", "--integration-method", type="string", dest="int_method",
               help="# Integration method: trapz or simps", metavar="STR")

op.set_defaults(fn = "no file here", temp = 298.15, nu_min = 0.e0, nu_max = 0.e0, nfree = 0.e0, neducts = 1, nproducts = 1, do_blc="false",int_method = "simps")

(opts, args) = op.parse_args ()
opts_dict = vars (opts)

print "#  _______ _                            _______        _ "
print "# |__   __| |                          |__   __|      (_)"
print "#    | |  | |__   ___ _ __ _ __ ___   ___ | |_ __ __ _ _ "
print "#    | |  | '_ \ / _ \ '__| '_ ` _ \ / _ \| | '__/ _` | |"
print "#    | |  | | | |  __/ |  | | | | | | (_) | | | | (_| | |"
print "#    |_|  |_| |_|\___|_|  |_| |_| |_|\___/|_|_|  \__,_| |"
print "#                                                    _/ |"
print "#                                                   |__/ "
print "#"
print "#"
print "# Version:              1.0 (March 2019)"
print "# Authors:              Laurens Peters, Johannes Dietschreit"
print "# Scientific Advisor:   Christian Ochsenfeld"
print "# Affiliation:          University of Munich (LMU)"
print "# Temperature:          %.2f" % opts.temp
if opts.nu_min != 0:
  print "# Start Wvn:            %.2f" % opts.nu_min
if opts.nu_max != 0:
  print "# Stop  Wvn:            %.2f" % opts.nu_max
print "# Num. Deg. of Freedom: %i" % opts.nfree
print "# Num. of Educts:       %i" % opts.neducts
print "# Num. of Products:     %i" % opts.nproducts

if opts.do_blc:
  print "# Baseline Correction:  ON"
else:
  print "# Baseline Correction:  OFF"

print "# Integration Method:   %s" % opts.int_method

the_data    = None
final_data  = None
final_data1 = None
deg_free    = None
energy      = None
entropy     = None
free        = None
fn          = args

# Constants and Variables

beta = 1.e0/(kb * opts.temp)    # in 1/a.u.

print "\n-- Degrees of Freedom --\n"

for l in range(0,len(args)):
  try:
  # Read Input
    final_data1 = final_data
    file1 = open (fn[l])
    lines1 = file1.readlines ()
    file1.close ()
    
  # Parse Data
    the_data = parse(opts, lines1)
    
  # Check Degrees of freedom

    deg_free = freedom(the_data, opts, beta)
    print "F(%i)       = %20.10e" % (l,deg_free)

  #Combine Data
    final_data = combine(final_data1, the_data, opts, l, deg_free)

  except:
    raise Exception ('Failed to read data from files!')

# Degrees of freedom
deg_free = freedom(final_data, opts, beta)

# Correction to energy

(energy_cl,energy_qm) = corr_energy(final_data, opts, beta)

# Correction to entropy

(entropy_cl,entropy_qm) = corr_entropy(final_data, opts, beta)

# Correction to free energy

(free_cl,free_qm) = corr_free(final_data, opts, beta)

# Print the results

print "F(tot)     = %20.10e"              % (deg_free)

print "\n-- Energy --\n"
print "E(classic) = %20.10e kJ/mol"       % (energy_cl*2625.49962e0)
print "E(quantum) = %20.10e kJ/mol"       % (energy_qm*2625.49962e0)
print "dE         = %20.10e kJ/mol"       % ((energy_qm-energy_cl)*2625.49962e0)

print "\n-- Entropy --\n"
print "S(classic) = %20.10e kJ/(mol x K)" % (entropy_cl*2625.49962e0)
print "S(quantum) = %20.10e kJ/(mol x K)" % (entropy_qm*2625.49962e0)
print "dS         = %20.10e kJ/(mol x K)" % ((entropy_qm-entropy_cl)*2625.49962e0)

print "\n-- Free Energy --\n"
print "A(classic) = %20.10e kJ/mol"       % (free_cl*2625.49962e0)
print "A(quantum) = %20.10e kJ/mol"       % (free_qm*2625.49962e0)
print "dA         = %20.10e kJ/mol"       % ((free_qm-free_cl)*2625.49962e0)

