#!/usr/bin/env python

import sys
from optparse import OptionParser
import numpy
import numpy.fft
import math
from scipy import signal
from scipy import integrate

# Constants and Variables

kb   = 3.166815343e-6           # Boltzmann in a.u./K
c    = 2.99792458e+10           # Speed of light in cm/s
h    = 6.626070040e-34/4.35974417e-18   # Planck constant in a.u. * s

# Parser:
# Format of velocity files:
# Time [fs] vel_x [sqrt(2*Hartree/amu)] vel_y [sqrt(2*Hartree/amu)] vel_z [sqrt(2*Hartree/amu)] mass [amu]

def parse(opts, lines):

  the_data = []
  for time,l in enumerate(lines):
    if time > opts.start:
      if not l.startswith("#"):
        try:
          vals = l.strip().split()
        except:
          raise Exception ("Failed to parse line...")

        try:
          timein = float(vals[0])
          valin1 = float(vals[1])
          valin2 = float(vals[2])
          valin3 = float(vals[3])
          valin4 = float(vals[4])

        except:
          raise Exception ("Failed to convert data... ")

        the_data.append ([timein, valin1, valin2, valin3, valin4]) #x,y,z,mass

  ndata = len(the_data)

  if ndata <= 0:
      raise Exception ("Failed to find any data, check target and options.")

  return the_data[0:opts.end]

# Get Autocorrelation Function

def autocorr(deriv_data, opts):

  autocorr_data = []
  np_data = numpy.einsum('ij->ji',numpy.asarray(deriv_data))
  npts = np_data.shape[1]
  dt = (np_data[0,1] - np_data[0,0]) #fs

  autox_in = np_data[1,:] - np_data[1,:].mean()
  autoy_in = np_data[2,:] - np_data[2,:].mean()
  autoz_in = np_data[3,:] - np_data[3,:].mean()
  mass     = np_data[4,:] 

  autox = signal.fftconvolve(autox_in, autox_in[::-1], mode='full')[-npts:]
  autoy = signal.fftconvolve(autoy_in, autoy_in[::-1], mode='full')[-npts:]
  autoz = signal.fftconvolve(autoz_in, autoz_in[::-1], mode='full')[-npts:]

  resx = autox/(1.e0*(numpy.arange(npts, 0, -1)))
  resy = autoy/(1.e0*(numpy.arange(npts, 0, -1)))
  resz = autoz/(1.e0*(numpy.arange(npts, 0, -1)))

  for l in range(0,len(autox)-opts.nlags):
    autocorr_data.append([l*dt,mass[l] * resx[l],mass[l] * resy[l],mass[l] * resz[l],mass[l]])

  return autocorr_data

# FFT

def fft_data(the_data, opts, lines, args):

  np_data = numpy.einsum('ij->ji',numpy.asarray(the_data))
  npts = np_data.shape[1]

  # Prepare x-Axis
  dt = (np_data[0,1] - np_data[0,0])  #keep in fs /0.02418884326505e0
  period = (npts-1)*dt - np_data[0,0] #keep in fs /0.02418884326505e0
  dw = 33356.40952 / period
  wmin = 0.0e0
  wmax = float(npts)*dw/2.0
  w = numpy.linspace(wmin, wmax, npts)

  # Damping

  if opts.do_damp == "true" or opts.do_damp == "yes":
    t0 = np_data[0,0]# / 0.02418884326505e0;
    ts = [math.exp(-(ttt-t0)/opts.damp_fac) for ttt in np_data[0,:] ] # math.exp(-(np_data[0,:] - t0)/opts.damp_fac);

    vsx = [a*b for a,b in zip(np_data[1,:],ts)]
    vsy = [a*b for a,b in zip(np_data[2,:],ts)]
    vsz = [a*b for a,b in zip(np_data[3,:],ts)]
    np_data = numpy.array([np_data[0,:],vsx,vsy,vsz,np_data[4,:]])

  # Prepare Input

  fft_in = (np_data[1,:] + np_data[2,:] + np_data[3,:])
  fft_in = numpy.concatenate((fft_in[len(fft_in)-1:0:-1],fft_in))

  # Fourier Transform

  np_fft = numpy.fft.fft(fft_in)
  np_fft *= dt/(1.0e15)

  # Prepare Spectra

  mask = numpy.zeros(shape=(len(np_fft),), dtype=float)
  mask[:] = 1.0
  mask[1::2] -= 2.0
  fw_pos = (mask*np_fft)[:npts] - (mask*np_fft)[::-1][:npts]
  
  fw_re = numpy.real(fw_pos)
  fw_im = numpy.imag(fw_pos)
  fw_abs = numpy.absolute(fw_pos)
  fw_re2 = numpy.real(fw_pos*fw_pos)

  # Norm

  norm =  1.e0/(kb * 298.15) * integrate.simps(fw_re[0:npts],dx = c*(w[1]-w[0]) )
  norm = 3.0 / norm
  print "# Norm: "+str(norm)

  # Return Spectrum

  return [w[0:npts],norm*fw_re[0:npts],norm*fw_im[0:npts],norm*fw_abs[0:npts],norm*fw_re2[0:npts]]

# Combine Results

def combine(final_data, specs, opts, loop):

  if loop == 0:
    final_data_new = specs
    for i, elements in enumerate(final_data_new):
      if i==0:
        elements *= 1.0/float(len(args))
      else:
        elements *= 1.0/float(opts.nrep)
  else:
    final_data_new = []
    for i in range(len(final_data)):
      final_data_new.append(final_data[i])
    for i, element in enumerate(specs):
      if i ==0:
        final_data_new[i] += 1.0/float(len(args)) * element
      else:
        final_data_new[i] += 1.0/float(opts.nrep) * element

  return final_data_new


op = OptionParser(usage="./SpecTraj.py velocity-files [options]\n\n"+
                        "create spectra from trajectories...\n")

op.add_option ("-b", "--start-time", type="int", dest="start",
               help="# start time [number of steps]", metavar="INT")
op.add_option ("-e", "--end-time", type="int", dest="end",
               help="# end time [number of steps]", metavar="INT")
op.add_option ("-n", "--nreps", type="int", dest="nrep",
               help="# number of replicas", metavar="INT")
op.add_option ("-d", "--damping", type="string", dest="do_damp",
               help="# use exponential damping", metavar="STR")
op.add_option ("-f", "--damping-factor", type="float", dest="damp_fac",
               help="# exponential damping factor", metavar="DOUBLE")
op.add_option ("-l", "--lags", type="int", dest="nlags",
               help="# reduce number of lags in autocorr", metavar="INT")

op.set_defaults(fn = "no file here", do_damp = "true", damp_fac=1000.e0, nlags=0, start=0, end=100000000, nrep=1)

(opts, args) = op.parse_args ()
opts_dict = vars (opts)


print "#   _____              _______        _ "
print "#  / ____|            |__   __|      (_)"
print "# | (___  _ __   ___  ___| |_ __ __ _ _ " 
print "#  \___ \| '_ \ / _ \/ __| | '__/ _` | |"
print "#  ____) | |_) |  __/ (__| | | | (_| | |"
print "# |_____/| .__/ \___|\___|_|_|  \__,_| |"
print "#        | |                        _/ |"
print "#        |_|                       |__/ "
print "#"
print "#"
print "# Version:            1.0 (March 2019)"
print "# Authors:            Laurens Peters, Johannes Dietschreit"
print "# Scientific Advisor: Christian Ochsenfeld"
print "# Affiliation:        University of Munich (LMU)"
print "# Start Time:         %.2f" % opts.start
print "# End Time:           %.2f" % opts.end
print "# Number of Replicas: %i" % opts.nrep
print "# Number of Lags:     %i" % opts.nlags

if opts.do_damp:
  print "# Exp. Damping:       ON"
  print "# Damping Factor:     %.2f" % opts.damp_fac
else:
  print "# Exp. Damping:       OFF"

the_data = None
autocorr_data = None
final_data = None
final_data1 = None
fn = args

for l in range(0,len(args)):
  try:
    final_data1 = final_data
  # Read Input
    file1 = open (fn[l])
    lines1 = file1.readlines ()
    file1.close ()

  # Parse Data
    the_data = parse(opts, lines1)

  # Get Autocorrelation
    autocorr_data = autocorr(the_data, opts)
    try:
      # Fourier Transform and plot
      spec = fft_data(autocorr_data, opts, lines1, len(args))
    except:
      raise Exception ('Failed to generate spectrum from file: '+'"'+fn+'"')

  # Combine Data
    final_data = combine(final_data1, spec, opts, l)

  except:
    raise Exception ('Failed to generate spectrum from file: '+'"'+fn+'"')


# Print Spectra

print "# Wavenr [cm-1] Real Imag Abs Abs^2"
for l in range(len(final_data[0])):
  print "%20.10e %20.10e %20.10e %20.10e %20.10e" % (final_data[0][l],final_data[1][l],final_data[2][l],final_data[3][l],final_data[4][l])


