/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.in;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.zip.GZIPInputStream;
import loci.common.DateTools;
import loci.common.Location;
import loci.common.RandomAccessInputStream;
import loci.formats.CoreMetadata;
import loci.formats.FormatException;
import loci.formats.FormatReader;
import loci.formats.FormatTools;
import loci.formats.MetadataTools;
import loci.formats.in.MetadataLevel;
import loci.formats.meta.MetadataStore;
import ome.units.UNITS;
import ome.units.quantity.Frequency;
import ome.units.quantity.Length;
import ome.units.quantity.Power;
import ome.units.quantity.Time;
import ome.xml.model.primitives.Timestamp;

public class ICSReader
extends FormatReader {
    public static final String NL = "\r\n";
    public static final String[] DATE_FORMATS = new String[]{"EEEE, MMMM dd, yyyy HH:mm:ss", "EEE dd MMMM yyyy HH:mm:ss", "EEE MMM dd HH:mm:ss yyyy", "EE dd MMM yyyy HH:mm:ss z", "HH:mm:ss dd\\MM\\yy"};
    private static final String[][] DOCUMENT_KEYS = new String[][]{{"date"}, {"document", "average"}, {"document"}, {"gmtdate"}, {"label"}};
    private static final String[][] HISTORY_KEYS = new String[][]{{"a\\d"}, {"acquisition", "acquire\\..*."}, {"acquisition", "laserbox\\..*."}, {"acquisition", "modules\\(.*."}, {"acquisition", "objective", "position"}, {"adc", "resolution"}, {"atd_hardware", "ver"}, {"atd_libraries", "ver"}, {"atd_microscopy", "ver"}, {"author"}, {"averagecount"}, {"averagequality"}, {"beam", "zoom"}, {"binning"}, {"bits/pixel"}, {"black", "level"}, {"black", "level\\*"}, {"black_level"}, {"camera", "manufacturer"}, {"camera", "model"}, {"camera"}, {"cfd", "holdoff"}, {"cfd", "limit", "high"}, {"cfd", "limit", "low"}, {"cfd", "zc", "level"}, {"channel\\*"}, {"collection", "time"}, {"cols"}, {"company"}, {"count", "increment"}, {"created", "on"}, {"creation", "date"}, {"cube", "descriptio"}, {"cube", "description"}, {"cube", "emm", "nm"}, {"cube", "exc", "nm"}, {"cube"}, {"date"}, {"dategmt"}, {"dead", "time", "comp"}, {"desc", "exc", "turret"}, {"desc", "emm", "turret"}, {"detector", "type"}, {"detector"}, {"dimensions"}, {"direct", "turret"}, {"dither", "range"}, {"dwell"}, {"excitationfwhm"}, {"experiment"}, {"experimenter"}, {"expon.", "order"}, {"exposure"}, {"exposure_time"}, {"ext", "latch", "delay"}, {"extents"}, {"filterset", "dichroic", "name"}, {"filterset", "dichroic", "nm"}, {"filterset", "emm", "name"}, {"filterset", "emm", "nm"}, {"filterset", "exc", "name"}, {"filterset", "exc", "nm"}, {"filterset"}, {"filter\\*"}, {"firmware"}, {"fret", "backgr\\d"}, {"frametime"}, {"gain"}, {"gain\\d"}, {"gain\\*"}, {"gamma"}, {"icsviewer", "ver"}, {"ht\\*"}, {"id"}, {"illumination", "mode", "laser"}, {"illumination", "mode"}, {"image", "bigendian"}, {"image", "bpp"}, {"image", "form"}, {"image", "physical_sizex"}, {"image", "physical_sizey"}, {"image", "sizex"}, {"image", "sizey"}, {"labels"}, {"lamp", "manufacturer"}, {"lamp", "model"}, {"laser", "firmware"}, {"laser", "manufacturer"}, {"laser", "model"}, {"laser", "power"}, {"laser", "rep", "rate"}, {"laser", "type"}, {"laser\\d", "intensity"}, {"laser\\d", "name"}, {"laser\\d", "wavelength"}, {"left"}, {"length"}, {"line", "compressio"}, {"line", "compression"}, {"linetime"}, {"magnification"}, {"manufacturer"}, {"max", "photon", "coun"}, {"max", "photon", "count"}, {"memory", "bank"}, {"metadata", "format", "ver"}, {"microscope", "built", "on"}, {"microscope", "name"}, {"microscope"}, {"mirror", "\\d"}, {"mode"}, {"noiseval"}, {"no.", "frames"}, {"objective", "detail"}, {"objective", "immersion"}, {"objective", "mag"}, {"objective", "magnification"}, {"objective", "na"}, {"objective", "type"}, {"objective", "workingdistance"}, {"objective"}, {"offsets"}, {"other", "text"}, {"passcount"}, {"pinhole"}, {"pixel", "clock"}, {"pixel", "time"}, {"pmt"}, {"polarity"}, {"region"}, {"rep", "period"}, {"repeat", "time"}, {"revision"}, {"routing", "chan", "x"}, {"routing", "chan", "y"}, {"rows"}, {"scan", "borders"}, {"scan", "flyback"}, {"scan", "pattern"}, {"scan", "pixels", "x"}, {"scan", "pixels", "y"}, {"scan", "pos", "x"}, {"scan", "pos", "y"}, {"scan", "resolution"}, {"scan", "speed"}, {"scan", "zoom"}, {"scanner", "lag"}, {"scanner", "pixel", "time"}, {"scanner", "resolution"}, {"scanner", "speed"}, {"scanner", "xshift"}, {"scanner", "yshift"}, {"scanner", "zoom"}, {"shutter\\d"}, {"shutter", "type"}, {"software"}, {"spectral", "bin_definition"}, {"spectral", "calibration", "gain", "data"}, {"spectral", "calibration", "gain", "mode"}, {"spectral", "calibration", "offset", "data"}, {"spectral", "calibration", "offset", "mode"}, {"spectral", "calibration", "sensitivity", "mode"}, {"spectral", "central_wavelength"}, {"spectral", "laser_shield"}, {"spectral", "laser_shield_width"}, {"spectral", "resolution"}, {"stage", "controller"}, {"stage", "firmware"}, {"stage", "manufacturer"}, {"stage", "model"}, {"stage", "pos"}, {"stage", "positionx"}, {"stage", "positiony"}, {"stage", "positionz"}, {"stage_xyzum"}, {"step\\d", "channel", "\\d"}, {"step\\d", "gain", "\\d"}, {"step\\d", "laser"}, {"step\\d", "name"}, {"step\\d", "pinhole"}, {"step\\d", "pmt", "ch", "\\d"}, {"step\\d", "shutter", "\\d"}, {"step\\d"}, {"stop", "on", "o'flow"}, {"stop", "on", "time"}, {"study"}, {"sync", "freq", "div"}, {"sync", "holdoff"}, {"sync"}, {"tac", "gain"}, {"tac", "limit", "low"}, {"tac", "offset"}, {"tac", "range"}, {"tau\\d"}, {"tcspc", "adc", "res"}, {"tcspc", "adc", "resolution"}, {"tcspc", "approx", "adc", "rate"}, {"tcspc", "approx", "cfd", "rate"}, {"tcspc", "approx", "tac", "rate"}, {"tcspc", "bh"}, {"tcspc", "cfd", "holdoff"}, {"tcspc", "cfd", "limit", "high"}, {"tcspc", "cfd", "limit", "low"}, {"tcspc", "cfd", "zc", "level"}, {"tcspc", "clock", "polarity"}, {"tcspc", "collection", "time"}, {"tcspc", "count", "increment"}, {"tcspc", "dead", "time", "enabled"}, {"tcspc", "delay"}, {"tcspc", "dither", "range"}, {"tcspc", "left", "border"}, {"tcspc", "line", "compression"}, {"tcspc", "mem", "offset"}, {"tcspc", "operation", "mode"}, {"tcspc", "overflow"}, {"tcspc", "pixel", "clk", "divider"}, {"tcspc", "pixel", "clock"}, {"tcspc", "routing", "x"}, {"tcspc", "routing", "y"}, {"tcspc", "scan", "x"}, {"tcspc", "scan", "y"}, {"tcspc", "sync", "divider"}, {"tcspc", "sync", "holdoff"}, {"tcspc", "sync", "rate"}, {"tcspc", "sync", "threshold"}, {"tcspc", "sync", "zc", "level"}, {"tcspc", "tac", "gain"}, {"tcspc", "tac", "limit", "high"}, {"tcspc", "tac", "limit", "low"}, {"tcspc", "tac", "offset"}, {"tcspc", "tac", "range"}, {"tcspc", "time", "window"}, {"tcspc", "top", "border"}, {"tcspc", "total", "frames"}, {"tcspc", "total", "time"}, {"tcspc", "trigger"}, {"tcspc", "x", "sync", "polarity"}, {"tcspc", "y", "sync", "polarity"}, {"text"}, {"time"}, {"title"}, {"top"}, {"transmission"}, {"trigger"}, {"type"}, {"units"}, {"version"}, {"wavelength\\*"}, {"x", "amplitude"}, {"y", "amplitude"}, {"x", "delay"}, {"y", "delay"}, {"x", "offset"}, {"y", "offset"}, {"z", "\\(background\\)"}};
    private static final String[][] LAYOUT_KEYS = new String[][]{{"coordinates"}, {"order"}, {"parameters"}, {"real_significant_bits"}, {"significant_bits"}, {"significant_channels"}, {"sizes"}};
    private static final String[][] PARAMETER_KEYS = new String[][]{{"allowedlinemodes"}, {"ch"}, {"higher_limit"}, {"labels"}, {"lower_limit"}, {"origin"}, {"range"}, {"sample_width", "ch"}, {"sample_width"}, {"scale"}, {"units", "adc-units", "channels"}, {"units", "adc-units", "nm"}, {"units"}};
    private static final String[][] REPRESENTATION_KEYS = new String[][]{{"byte_order"}, {"compression"}, {"format"}, {"sign"}};
    private static final String[][] SENSOR_KEYS = new String[][]{{"model"}, {"s_params", "channels"}, {"s_params", "exphotoncnt"}, {"s_params", "lambdaem"}, {"s_params", "lambdaex"}, {"s_params", "numaperture"}, {"s_params", "pinholeradius"}, {"s_params", "pinholespacing"}, {"s_params", "refinxlensmedium"}, {"s_params", "refinxmedium"}, {"s_params", "refrinxlensmedium"}, {"s_params", "refrinxmedium"}, {"type"}};
    private static final String[][] VIEW_KEYS = new String[][]{{"view", "color", "lib", "lut"}, {"view", "color", "count"}, {"view", "color", "doc", "scale"}, {"view", "color", "mode", "rgb", "set"}, {"view", "color", "mode", "rgb"}, {"view", "color", "schemes"}, {"view", "color", "view", "active"}, {"view", "color"}, {"view\\d", "alpha"}, {"view\\d", "alphastate"}, {"view\\d", "annotation", "annellipse"}, {"view\\d", "annotation", "annpoint"}, {"view\\d", "autoresize"}, {"view\\d", "axis"}, {"view\\d", "blacklevel"}, {"view\\d", "color"}, {"view\\d", "cursor"}, {"view\\d", "dimviewoption"}, {"view\\d", "gamma"}, {"view\\d", "ignoreaspect"}, {"view\\d", "intzoom"}, {"view\\d", "live"}, {"view\\d", "order"}, {"view\\d", "port"}, {"view\\d", "position"}, {"view\\d", "saturation"}, {"view\\d", "scale"}, {"view\\d", "showall"}, {"view\\d", "showcursor"}, {"view\\d", "showindex"}, {"view\\d", "size"}, {"view\\d", "synchronize"}, {"view\\d", "tile"}, {"view\\d", "useunits"}, {"view\\d", "zoom"}, {"view\\d"}, {"view"}};
    private static String[][] OTHER_KEYS = new String[][]{{"cube", "descriptio"}, {"cube", "description"}, {"image", "form"}, {"refinxlensmedium"}, {"refinxmedium"}, {"scil_type"}, {"source"}};
    private String currentIcsId;
    private String currentIdsId;
    private boolean versionTwo;
    private byte[] data;
    private long offset;
    private boolean gzip;
    private GZIPInputStream gzipStream;
    private boolean invertY;
    private boolean lifetime;
    private String labels;
    private Vector<Integer> channelLengths;
    private Vector<String> channelTypes;
    private int prevImage;
    private boolean hasInstrumentData = false;
    private boolean storedRGB = false;

    public ICSReader() {
        super("Image Cytometry Standard", new String[]{"ics", "ids"});
        this.domains = new String[]{"Light Microscopy", "Fluorescence-Lifetime Imaging", "Unknown"};
        this.hasCompanionFiles = true;
        this.datasetDescription = "One .ics and possibly one .ids with a similar name";
    }

    @Override
    public boolean isSingleFile(String id) throws FormatException, IOException {
        RandomAccessInputStream f = new RandomAccessInputStream(id);
        boolean singleFile = f.readString(17).trim().equals("ics_version\t2.0");
        f.close();
        return singleFile;
    }

    @Override
    public String[] getDomains() {
        FormatTools.assertId(this.currentId, true, 1);
        String[] domain = new String[]{"Graphics"};
        if (this.getModuloC().length() > 1) {
            domain[0] = "Fluorescence-Lifetime Imaging";
        } else if (this.hasInstrumentData) {
            domain[0] = "Light Microscopy";
        }
        return domain;
    }

    @Override
    public boolean isInterleaved(int subC) {
        FormatTools.assertId(this.currentId, true, 1);
        return subC == 0 && ((CoreMetadata)this.core.get((int)0)).interleaved;
    }

    @Override
    public int fileGroupOption(String id) throws FormatException, IOException {
        return 0;
    }

    @Override
    public byte[] openBytes(int no, byte[] buf, int x, int y, int w, int h) throws FormatException, IOException {
        int sizeC;
        FormatTools.checkPlaneParameters(this, no, buf.length, x, y, w, h);
        int bpp = FormatTools.getBytesPerPixel(this.getPixelType());
        int len = FormatTools.getPlaneSize(this);
        int pixel = bpp * this.getRGBChannelCount();
        int rowLen = FormatTools.getPlaneSize(this, w, 1);
        int[] coordinates = this.getZCTCoords(no);
        int[] prevCoordinates = this.getZCTCoords(this.prevImage);
        if (!this.gzip) {
            this.in.seek(this.offset + (long)no * (long)len);
        } else {
            long toSkip = (long)(no - this.prevImage - 1) * (long)len;
            if (this.gzipStream == null || no <= this.prevImage) {
                FileInputStream fis = null;
                toSkip = (long)no * (long)len;
                if (this.versionTwo) {
                    fis = new FileInputStream(this.currentIcsId);
                    fis.skip(this.offset);
                } else {
                    fis = new FileInputStream(this.currentIdsId);
                    toSkip += this.offset;
                }
                try {
                    this.gzipStream = new GZIPInputStream(fis);
                }
                catch (IOException e) {
                    this.gzip = false;
                    this.in.seek(this.offset + (long)no * (long)len);
                    this.gzipStream = null;
                }
            }
            if (this.gzipStream != null) {
                while (toSkip > 0L) {
                    toSkip -= this.gzipStream.skip(toSkip);
                }
                this.data = new byte[len * (this.storedRGB ? this.getSizeC() : 1)];
                for (int toRead = this.data.length; toRead > 0; toRead -= this.gzipStream.read(this.data, this.data.length - toRead, toRead)) {
                }
            }
        }
        int n = sizeC = this.lifetime ? 1 : this.getSizeC();
        if (!this.isRGB() && this.channelLengths.size() == 1 && this.storedRGB) {
            this.in.seek(this.offset + (long)len * (long)this.getIndex(coordinates[0], 0, coordinates[2]));
            if (!this.gzip && this.data == null) {
                this.data = new byte[len * this.getSizeC()];
                this.in.read(this.data);
            } else if (!(this.gzip || coordinates[0] == prevCoordinates[0] && coordinates[2] == prevCoordinates[2])) {
                this.in.read(this.data);
            }
            for (int row = y; row < h + y; ++row) {
                for (int col = x; col < w + x; ++col) {
                    int src = bpp * (no % this.getSizeC() + sizeC * (row * this.getSizeX() + col));
                    int dest = bpp * ((row - y) * w + (col - x));
                    System.arraycopy(this.data, src, buf, dest, bpp);
                }
            }
        } else if (this.gzip) {
            RandomAccessInputStream s = new RandomAccessInputStream(this.data);
            this.readPlane(s, x, y, w, h, buf);
            s.close();
        } else {
            this.readPlane(this.in, x, y, w, h, buf);
        }
        if (this.invertY) {
            byte[] row = new byte[rowLen];
            for (int r = 0; r < h / 2; ++r) {
                int topOffset = r * rowLen;
                int bottomOffset = (h - r - 1) * rowLen;
                System.arraycopy(buf, topOffset, row, 0, rowLen);
                System.arraycopy(buf, bottomOffset, buf, topOffset, rowLen);
                System.arraycopy(row, 0, buf, bottomOffset, rowLen);
            }
        }
        this.prevImage = no;
        return buf;
    }

    @Override
    public String[] getSeriesUsedFiles(boolean noPixels) {
        String[] stringArray;
        FormatTools.assertId(this.currentId, true, 1);
        if (this.versionTwo) {
            String[] stringArray2;
            if (noPixels) {
                stringArray2 = null;
            } else {
                String[] stringArray3 = new String[1];
                stringArray2 = stringArray3;
                stringArray3[0] = this.currentIcsId;
            }
            return stringArray2;
        }
        if (noPixels) {
            String[] stringArray4 = new String[1];
            stringArray = stringArray4;
            stringArray4[0] = this.currentIcsId;
        } else {
            String[] stringArray5 = new String[2];
            stringArray5[0] = this.currentIcsId;
            stringArray = stringArray5;
            stringArray5[1] = this.currentIdsId;
        }
        return stringArray;
    }

    @Override
    public void close(boolean fileOnly) throws IOException {
        super.close(fileOnly);
        if (!fileOnly) {
            this.currentIcsId = null;
            this.currentIdsId = null;
            this.data = null;
            this.versionTwo = false;
            this.gzip = false;
            this.invertY = false;
            this.lifetime = false;
            this.prevImage = 0;
            this.hasInstrumentData = false;
            this.storedRGB = false;
            if (this.gzipStream != null) {
                this.gzipStream.close();
            }
            this.gzipStream = null;
        }
    }

    @Override
    public void reopenFile() throws IOException {
        if (this.in != null) {
            this.in.close();
        }
        this.in = this.versionTwo ? new RandomAccessInputStream(this.currentIcsId) : new RandomAccessInputStream(this.currentIdsId);
        this.in.order(this.isLittleEndian());
    }

    @Override
    protected void initFile(String id) throws FormatException, IOException {
        char[] c;
        String ext;
        super.initFile(id);
        LOGGER.info("Finding companion file");
        String icsId = id;
        String idsId = id;
        int dot = id.lastIndexOf(".");
        String string = ext = dot < 0 ? "" : id.substring(dot + 1).toLowerCase();
        if (ext.equals("ics")) {
            c = idsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] + '\u0001');
            idsId = new String(c);
        } else if (ext.equals("ids")) {
            c = icsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] - '\u0001');
            icsId = new String(c);
        }
        if (icsId == null) {
            throw new FormatException("No ICS file found.");
        }
        Location icsFile = new Location(icsId);
        if (!icsFile.exists()) {
            throw new FormatException("ICS file not found.");
        }
        LOGGER.info("Checking file version");
        RandomAccessInputStream f = new RandomAccessInputStream(icsId);
        if (f.readString(17).trim().equals("ics_version\t2.0")) {
            this.in = new RandomAccessInputStream(icsId);
            this.versionTwo = true;
        } else {
            if (idsId == null) {
                throw new FormatException("No IDS file found.");
            }
            Location idsFile = new Location(idsId);
            if (!idsFile.exists()) {
                throw new FormatException("IDS file not found.");
            }
            this.currentIdsId = idsId;
            this.in = new RandomAccessInputStream(this.currentIdsId);
        }
        f.close();
        this.currentIcsId = icsId;
        LOGGER.info("Reading metadata");
        CoreMetadata m = (CoreMetadata)this.core.get(0);
        Double[] scales = null;
        Double[] timestamps = null;
        String[] units = null;
        String[] axes = null;
        int[] axisLengths = null;
        String byteOrder = null;
        String rFormat = null;
        String compression = null;
        RandomAccessInputStream reader = new RandomAccessInputStream(icsId);
        reader.seek(0L);
        reader.readString(NL);
        String line = reader.readString(NL);
        boolean signed = false;
        StringBuilder textBlock = new StringBuilder();
        double[] sizes = null;
        Double[] emWaves = null;
        Double[] exWaves = null;
        Length[] stagePos = null;
        String imageName = null;
        String date = null;
        Object description = null;
        Double magnification = null;
        Double lensNA = null;
        Double workingDistance = null;
        Object objectiveModel = null;
        Object immersion = null;
        Object lastName = null;
        Hashtable<Integer, Double> gains = new Hashtable<Integer, Double>();
        Hashtable<Integer, Double> pinholes = new Hashtable<Integer, Double>();
        Hashtable<Integer, Double> wavelengths = new Hashtable<Integer, Double>();
        Hashtable<Integer, Object> channelNames = new Hashtable<Integer, Object>();
        Object laserModel = null;
        Object laserManufacturer = null;
        Double laserPower = null;
        Double laserRepetitionRate = null;
        Object detectorManufacturer = null;
        Object detectorModel = null;
        Object microscopeModel = null;
        Object microscopeManufacturer = null;
        Iterator<String> experimentType = null;
        Time exposureTime = null;
        Object filterSetModel = null;
        Object dichroicModel = null;
        Object excitationModel = null;
        Object emissionModel = null;
        while (line != null && !line.trim().equals("end") && reader.getFilePointer() < reader.length() - 1L) {
            if ((line = line.trim()).length() > 0) {
                Object value;
                String key;
                String[] tokens = this.tokenize(line);
                String token0 = tokens[0].toLowerCase();
                String[] keyValue = null;
                if (token0.equals("ics_version")) {
                    String value2 = this.concatenateTokens(tokens, 1, tokens.length);
                    this.addGlobalMeta(token0, value2);
                } else if (token0.equals("filename")) {
                    imageName = this.concatenateTokens(tokens, 1, tokens.length);
                    this.addGlobalMeta(token0, imageName);
                } else if (token0.equals("layout")) {
                    keyValue = this.findKeyValue(tokens, LAYOUT_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    if (key.equalsIgnoreCase("layout sizes")) {
                        StringTokenizer t = new StringTokenizer((String)value);
                        axisLengths = new int[t.countTokens()];
                        for (int n = 0; n < axisLengths.length; ++n) {
                            try {
                                axisLengths[n] = Integer.parseInt(t.nextToken().trim());
                                continue;
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse axis length", e);
                            }
                        }
                    } else if (key.equalsIgnoreCase("layout order")) {
                        StringTokenizer t = new StringTokenizer((String)value);
                        axes = new String[t.countTokens()];
                        for (int n = 0; n < axes.length; ++n) {
                            axes[n] = t.nextToken().trim();
                        }
                    } else if (key.equalsIgnoreCase("layout significant_bits")) {
                        m.bitsPerPixel = Integer.parseInt((String)value);
                    }
                } else if (token0.equals("representation")) {
                    keyValue = this.findKeyValue(tokens, REPRESENTATION_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    if (key.equalsIgnoreCase("representation byte_order")) {
                        byteOrder = value;
                    } else if (key.equalsIgnoreCase("representation format")) {
                        rFormat = value;
                    } else if (key.equalsIgnoreCase("representation compression")) {
                        compression = value;
                    } else if (key.equalsIgnoreCase("representation sign")) {
                        signed = ((String)value).equals("signed");
                    }
                } else if (token0.equals("parameter")) {
                    keyValue = this.findKeyValue(tokens, PARAMETER_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    if (key.equalsIgnoreCase("parameter scale")) {
                        scales = this.splitDoubles((String)value);
                    } else if (key.equalsIgnoreCase("parameter t")) {
                        timestamps = this.splitDoubles((String)value);
                    } else if (key.equalsIgnoreCase("parameter units")) {
                        units = ((String)value).split("\\s+");
                    }
                    if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM && key.equalsIgnoreCase("parameter ch")) {
                        String[] names = ((String)value).split(" ");
                        for (int n = 0; n < names.length; ++n) {
                            channelNames.put(new Integer(n), names[n].trim());
                        }
                    }
                } else if (token0.equals("history")) {
                    keyValue = this.findKeyValue(tokens, HISTORY_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    Double doubleValue = null;
                    try {
                        doubleValue = new Double((String)value);
                    }
                    catch (NumberFormatException e) {
                        LOGGER.debug("Could not parse double value '{}'", value, (Object)e);
                    }
                    if (key.equalsIgnoreCase("history software") && ((String)value).indexOf("SVI") != -1) {
                        this.invertY = true;
                    } else if (key.equalsIgnoreCase("history date") || key.equalsIgnoreCase("history created on")) {
                        if (((String)value).indexOf(32) > 0) {
                            date = ((String)value).substring(0, ((String)value).lastIndexOf(" "));
                            date = DateTools.formatDate(date, DATE_FORMATS);
                        }
                    } else if (key.equalsIgnoreCase("history creation date")) {
                        date = DateTools.formatDate(value, DATE_FORMATS);
                    } else if (key.equalsIgnoreCase("history type")) {
                        if (((String)value).equalsIgnoreCase("time resolved") || ((String)value).equalsIgnoreCase("FluorescenceLifetime")) {
                            this.lifetime = true;
                        }
                        experimentType = value;
                    } else if (key.equalsIgnoreCase("history labels")) {
                        this.labels = value;
                    } else if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
                        if (key.equalsIgnoreCase("history") || key.equalsIgnoreCase("history text")) {
                            textBlock.append((String)value);
                            textBlock.append("\n");
                            this.metadata.remove(key);
                        } else if (key.startsWith("history gain")) {
                            Integer n = 0;
                            try {
                                n = new Integer(key.substring(12).trim());
                                n = new Integer(n - 1);
                            }
                            catch (NumberFormatException e) {
                                // empty catch block
                            }
                            if (doubleValue != null) {
                                gains.put(n, doubleValue);
                            }
                        } else if (key.startsWith("history laser") && key.endsWith("wavelength")) {
                            int laser = Integer.parseInt(key.substring(13, key.indexOf(" ", 13))) - 1;
                            value = ((String)value).replaceAll("nm", "").trim();
                            try {
                                wavelengths.put(new Integer(laser), new Double((String)value));
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse wavelength", e);
                            }
                        } else if (key.equalsIgnoreCase("history Wavelength*")) {
                            String[] waves = ((String)value).split(" ");
                            for (int i = 0; i < waves.length; ++i) {
                                wavelengths.put(new Integer(i), new Double(waves[i]));
                            }
                        } else if (key.equalsIgnoreCase("history laser manufacturer")) {
                            laserManufacturer = value;
                        } else if (key.equalsIgnoreCase("history laser model")) {
                            laserModel = value;
                        } else if (key.equalsIgnoreCase("history laser power")) {
                            try {
                                laserPower = new Double((String)value);
                            }
                            catch (NumberFormatException waves) {}
                        } else if (key.equalsIgnoreCase("history laser rep rate")) {
                            Object repRate = value;
                            if (((String)repRate).indexOf(32) != -1) {
                                repRate = ((String)repRate).substring(0, ((String)repRate).lastIndexOf(" "));
                            }
                            laserRepetitionRate = new Double((String)repRate);
                        } else if (key.equalsIgnoreCase("history objective type") || key.equalsIgnoreCase("history objective")) {
                            objectiveModel = value;
                        } else if (key.equalsIgnoreCase("history objective immersion")) {
                            immersion = value;
                        } else if (key.equalsIgnoreCase("history objective NA")) {
                            lensNA = doubleValue;
                        } else if (key.equalsIgnoreCase("history objective WorkingDistance")) {
                            workingDistance = doubleValue;
                        } else if (key.equalsIgnoreCase("history objective magnification") || key.equalsIgnoreCase("history objective mag")) {
                            magnification = doubleValue;
                        } else if (key.equalsIgnoreCase("history camera manufacturer")) {
                            detectorManufacturer = value;
                        } else if (key.equalsIgnoreCase("history camera model")) {
                            detectorModel = value;
                        } else if (key.equalsIgnoreCase("history author") || key.equalsIgnoreCase("history experimenter")) {
                            lastName = value;
                        } else if (key.equalsIgnoreCase("history extents")) {
                            String[] lengths = ((String)value).split(" ");
                            sizes = new double[lengths.length];
                            for (int n = 0; n < sizes.length; ++n) {
                                try {
                                    sizes[n] = Double.parseDouble(lengths[n].trim());
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse axis length", e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("history stage_xyzum")) {
                            String[] positions = ((String)value).split(" ");
                            stagePos = new Length[positions.length];
                            for (int n = 0; n < stagePos.length; ++n) {
                                try {
                                    Double number = Double.valueOf(positions[n]);
                                    stagePos[n] = new Length(number, UNITS.REFERENCEFRAME);
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse stage position", e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("history stage positionx")) {
                            if (stagePos == null) {
                                stagePos = new Length[3];
                            }
                            Double number = Double.valueOf((String)value);
                            stagePos[0] = new Length(number, UNITS.REFERENCEFRAME);
                        } else if (key.equalsIgnoreCase("history stage positiony")) {
                            if (stagePos == null) {
                                stagePos = new Length[3];
                            }
                            Double number = Double.valueOf((String)value);
                            stagePos[1] = new Length(number, UNITS.REFERENCEFRAME);
                        } else if (key.equalsIgnoreCase("history stage positionz")) {
                            if (stagePos == null) {
                                stagePos = new Length[3];
                            }
                            Double number = Double.valueOf((String)value);
                            stagePos[2] = new Length(number, UNITS.REFERENCEFRAME);
                        } else if (key.equalsIgnoreCase("history other text")) {
                            description = value;
                        } else if (key.startsWith("history step") && key.endsWith("name")) {
                            Integer n = new Integer(key.substring(12, key.indexOf(" ", 12)));
                            channelNames.put(n, value);
                        } else if (key.equalsIgnoreCase("history cube")) {
                            channelNames.put(new Integer(channelNames.size()), value);
                        } else if (key.equalsIgnoreCase("history cube emm nm")) {
                            if (emWaves == null) {
                                emWaves = new Double[]{new Double(((String)value).split(" ")[1].trim())};
                            }
                        } else if (key.equalsIgnoreCase("history cube exc nm")) {
                            if (exWaves == null) {
                                exWaves = new Double[]{new Double(((String)value).split(" ")[1].trim())};
                            }
                        } else if (key.equalsIgnoreCase("history microscope")) {
                            microscopeModel = value;
                        } else if (key.equalsIgnoreCase("history manufacturer")) {
                            microscopeManufacturer = value;
                        } else if (key.equalsIgnoreCase("history Exposure")) {
                            Double expDouble;
                            Object expTime = value;
                            if (((String)expTime).indexOf(32) != -1) {
                                expTime = ((String)expTime).substring(0, ((String)expTime).indexOf(32));
                            }
                            if ((expDouble = new Double((String)expTime)) != null) {
                                exposureTime = new Time(expDouble, UNITS.SECOND);
                            }
                        } else if (key.equalsIgnoreCase("history filterset")) {
                            filterSetModel = value;
                        } else if (key.equalsIgnoreCase("history filterset dichroic name")) {
                            dichroicModel = value;
                        } else if (key.equalsIgnoreCase("history filterset exc name")) {
                            excitationModel = value;
                        } else if (key.equalsIgnoreCase("history filterset emm name")) {
                            emissionModel = value;
                        }
                    }
                } else if (token0.equals("document")) {
                    keyValue = this.findKeyValue(tokens, DOCUMENT_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                } else if (token0.equals("sensor")) {
                    keyValue = this.findKeyValue(tokens, SENSOR_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
                        if (key.equalsIgnoreCase("sensor s_params LambdaEm")) {
                            String[] waves = ((String)value).split(" ");
                            emWaves = new Double[waves.length];
                            for (int n = 0; n < emWaves.length; ++n) {
                                try {
                                    emWaves[n] = new Double(Double.parseDouble(waves[n]));
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse emission wavelength", e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("sensor s_params LambdaEx")) {
                            String[] waves = ((String)value).split(" ");
                            exWaves = new Double[waves.length];
                            for (int n = 0; n < exWaves.length; ++n) {
                                try {
                                    exWaves[n] = new Double(Double.parseDouble(waves[n]));
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse excitation wavelength", e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("sensor s_params PinholeRadius")) {
                            String[] pins = ((String)value).split(" ");
                            int channel = 0;
                            for (int n = 0; n < pins.length; ++n) {
                                if (pins[n].trim().equals("")) continue;
                                try {
                                    pinholes.put(new Integer(channel++), new Double(pins[n]));
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse pinhole", e);
                                }
                            }
                        }
                    }
                } else if (token0.equals("view")) {
                    keyValue = this.findKeyValue(tokens, VIEW_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    if (key.equalsIgnoreCase("view view color lib lut")) {
                        int redIndex = ((String)value).toLowerCase().lastIndexOf("red");
                        int greenIndex = ((String)value).toLowerCase().lastIndexOf("green");
                        int blueIndex = ((String)value).toLowerCase().lastIndexOf("blue");
                        int index = redIndex > 0 && redIndex > greenIndex && redIndex > blueIndex ? redIndex + "red".length() : (greenIndex > 0 && greenIndex > redIndex && greenIndex > blueIndex ? greenIndex + "green".length() : (blueIndex > 0 && blueIndex > redIndex && blueIndex > greenIndex ? blueIndex + "blue".length() : ((String)value).indexOf(32)));
                        if (index > 0) {
                            key = key + ' ' + ((String)value).substring(0, index);
                            value = ((String)value).substring(index + 1);
                        }
                    } else if (key.equalsIgnoreCase("view view color mode rgb set")) {
                        int index = ((String)value).toLowerCase().lastIndexOf("colors");
                        index = index > 0 ? (index += "colors".length()) : ((String)value).indexOf(32);
                        if (index > 0) {
                            key = key + ' ' + ((String)value).substring(0, index);
                            value = ((String)value).substring(index + 1);
                        }
                    }
                    this.addGlobalMeta(key, value);
                } else {
                    LOGGER.debug("Unknown category " + token0);
                }
            }
            line = reader.readString(NL);
        }
        reader.close();
        this.hasInstrumentData = emWaves != null || exWaves != null || lensNA != null || stagePos != null || magnification != null || workingDistance != null || objectiveModel != null || immersion != null;
        this.addGlobalMeta("history text", textBlock.toString());
        LOGGER.info("Populating core metadata");
        m.rgb = false;
        m.dimensionOrder = "XY";
        this.channelLengths = new Vector();
        this.channelTypes = new Vector();
        int bitsPerPixel = 0;
        for (int i = 0; i < axes.length && i < axisLengths.length; ++i) {
            if (axes[i].equals("bits")) {
                bitsPerPixel = axisLengths[i];
                while (bitsPerPixel % 8 != 0) {
                    ++bitsPerPixel;
                }
                if (bitsPerPixel != 24 && bitsPerPixel != 48) continue;
                bitsPerPixel /= 3;
                continue;
            }
            if (axes[i].equals("x")) {
                m.sizeX = axisLengths[i];
                continue;
            }
            if (axes[i].equals("y")) {
                m.sizeY = axisLengths[i];
                continue;
            }
            if (axes[i].equals("z")) {
                m.sizeZ = axisLengths[i];
                if (this.getDimensionOrder().indexOf(90) != -1) continue;
                m.dimensionOrder = m.dimensionOrder + 'Z';
                continue;
            }
            if (axes[i].equals("t")) {
                m.sizeT = this.getSizeT() == 0 ? axisLengths[i] : (m.sizeT *= axisLengths[i]);
                if (this.getDimensionOrder().indexOf(84) != -1) continue;
                m.dimensionOrder = m.dimensionOrder + 'T';
                continue;
            }
            m.sizeC = m.sizeC == 0 ? axisLengths[i] : (m.sizeC *= axisLengths[i]);
            this.channelLengths.add(new Integer(axisLengths[i]));
            this.storedRGB = this.getSizeX() == 0;
            boolean bl = m.rgb = this.getSizeX() == 0 && this.getSizeC() <= 4 && this.getSizeC() > 1;
            if (this.getDimensionOrder().indexOf(67) == -1) {
                m.dimensionOrder = m.dimensionOrder + 'C';
            }
            if (axes[i].startsWith("c")) {
                this.channelTypes.add("Channel");
                continue;
            }
            if (axes[i].equals("p")) {
                this.channelTypes.add("Phase");
                continue;
            }
            if (axes[i].equals("f")) {
                this.channelTypes.add("Frequency");
                continue;
            }
            this.channelTypes.add("");
        }
        if (this.channelLengths.isEmpty()) {
            this.channelLengths.add(1);
            this.channelTypes.add("Channel");
        }
        if (this.isRGB() && emWaves != null && emWaves.length == this.getSizeC()) {
            m.rgb = false;
            this.storedRGB = true;
        }
        m.dimensionOrder = MetadataTools.makeSaneDimensionOrder(this.getDimensionOrder());
        if (this.getSizeZ() == 0) {
            m.sizeZ = 1;
        }
        if (this.getSizeC() == 0) {
            m.sizeC = 1;
        }
        if (this.getSizeT() == 0) {
            m.sizeT = 1;
        }
        if (this.channelLengths.size() > 0) {
            int clen0 = this.channelLengths.get(0);
            String ctype0 = this.channelTypes.get(0);
            boolean same = true;
            for (Integer len : this.channelLengths) {
                if (clen0 == len) continue;
                same = false;
            }
            for (String type : this.channelTypes) {
                if (ctype0.equals(type)) continue;
                same = false;
            }
            if (same) {
                m.moduloC.type = ctype0;
                if ("Lifetime".equals(ctype0)) {
                    m.moduloC.parentType = "Spectra";
                }
                m.moduloC.typeDescription = "TCSPC";
                m.moduloC.start = 0.0;
                m.moduloC.step = 1.0;
                m.moduloC.end = clen0 - 1;
            }
        }
        m.interleaved = this.isRGB();
        m.indexed = false;
        m.falseColor = false;
        m.metadataComplete = true;
        m.littleEndian = true;
        if (this.lifetime && this.labels != null) {
            int binCount = 0;
            String newOrder = null;
            if (this.labels.equalsIgnoreCase("t x y")) {
                newOrder = "XYCZT";
                m.interleaved = true;
                binCount = m.sizeX;
                m.sizeX = m.sizeY;
                m.sizeY = m.sizeZ;
                m.sizeZ = 1;
            } else if (this.labels.equalsIgnoreCase("x y t")) {
                newOrder = "XYCZT";
                binCount = m.sizeZ;
                m.sizeZ = 1;
            } else {
                LOGGER.debug("Lifetime data, unexpected 'history labels' " + this.labels);
            }
            if (newOrder != null) {
                m.dimensionOrder = newOrder;
                m.sizeC = binCount;
                m.moduloC.parentType = "Lifetime";
            }
        }
        m.imageCount = this.getSizeZ() * this.getSizeT();
        if (!this.isRGB()) {
            m.imageCount *= this.getSizeC();
        }
        if (byteOrder != null) {
            String firstByte = byteOrder.split(" ")[0];
            int first = Integer.parseInt(firstByte);
            m.littleEndian = rFormat.equals("real") ? first == 1 : first != 1;
        }
        boolean bl = this.gzip = compression == null ? false : compression.equals("gzip");
        if (this.versionTwo) {
            String s = this.in.readString(NL);
            while (!s.trim().equals("end")) {
                s = this.in.readString(NL);
            }
        }
        this.offset = this.in.getFilePointer();
        int bytes = bitsPerPixel / 8;
        if (bitsPerPixel < 32) {
            m.littleEndian = !this.isLittleEndian();
        }
        boolean fp = rFormat.equals("real");
        m.pixelType = FormatTools.pixelTypeFromBytes(bytes, signed, fp);
        LOGGER.info("Populating OME metadata");
        MetadataStore store = this.makeFilterMetadata();
        MetadataTools.populatePixels(store, this, true);
        store.setImageName(imageName, 0);
        if (date != null) {
            store.setImageAcquisitionDate(new Timestamp(date), 0);
        }
        if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
            store.setImageDescription((String)description, 0);
            String instrumentID = MetadataTools.createLSID("Instrument", 0);
            store.setInstrumentID(instrumentID, 0);
            store.setMicroscopeModel((String)microscopeModel, 0);
            store.setMicroscopeManufacturer((String)microscopeManufacturer, 0);
            store.setImageInstrumentRef(instrumentID, 0);
            store.setExperimentID(MetadataTools.createLSID("Experiment", 0), 0);
            store.setExperimentType(this.getExperimentType((String)((Object)experimentType)), 0);
            if (scales != null) {
                if (units != null && units.length == scales.length - 1) {
                    ArrayList<String> realUnits = new ArrayList<String>();
                    int unitIndex = 0;
                    for (int i = 0; i < axes.length; ++i) {
                        if (axes[i].toLowerCase().equals("ch")) {
                            realUnits.add("nm");
                            continue;
                        }
                        realUnits.add(units[unitIndex++]);
                    }
                    units = realUnits.toArray(new String[realUnits.size()]);
                }
                for (int i = 0; i < scales.length; ++i) {
                    String unit;
                    Double scale = scales[i];
                    if (scale == null) continue;
                    String axis = axes != null && axes.length > i ? axes[i] : "";
                    String string2 = unit = units != null && units.length > i ? units[i] : "";
                    if (axis.equals("x")) {
                        Length x;
                        if (!this.checkUnit(unit, "um", "microns", "micrometers") || (x = FormatTools.getPhysicalSizeX(scale)) == null) continue;
                        store.setPixelsPhysicalSizeX(x, 0);
                        continue;
                    }
                    if (axis.equals("y")) {
                        Length y;
                        if (!this.checkUnit(unit, "um", "microns", "micrometers") || (y = FormatTools.getPhysicalSizeY(scale)) == null) continue;
                        store.setPixelsPhysicalSizeY(y, 0);
                        continue;
                    }
                    if (axis.equals("z")) {
                        Length z;
                        if (!this.checkUnit(unit, "um", "microns", "micrometers") || (z = FormatTools.getPhysicalSizeZ(scale)) == null) continue;
                        store.setPixelsPhysicalSizeZ(z, 0);
                        continue;
                    }
                    if (!axis.equals("t") || scale == null) continue;
                    if (this.checkUnit(unit, "ms")) {
                        store.setPixelsTimeIncrement(new Time(scale, UNITS.MILLISECOND), 0);
                        continue;
                    }
                    if (!this.checkUnit(unit, "seconds") && !this.checkUnit(unit, "s")) continue;
                    store.setPixelsTimeIncrement(new Time(scale, UNITS.SECOND), 0);
                }
            } else if (sizes != null) {
                Length x;
                if (sizes.length > 0 && (x = FormatTools.getPhysicalSizeX(sizes[0])) != null) {
                    store.setPixelsPhysicalSizeX(x, 0);
                }
                if (sizes.length > 1) {
                    sizes[1] = sizes[1] / (double)this.getSizeY();
                    Length y = FormatTools.getPhysicalSizeY(sizes[1]);
                    if (y != null) {
                        store.setPixelsPhysicalSizeY(y, 0);
                    }
                }
            }
            if (timestamps != null) {
                for (int t = 0; t < timestamps.length && t < this.getSizeT(); ++t) {
                    Time deltaT;
                    if (timestamps[t] == null || Double.isNaN((deltaT = new Time(timestamps[t], UNITS.SECOND)).value().doubleValue())) continue;
                    for (int z = 0; z < this.getSizeZ(); ++z) {
                        for (int c2 = 0; c2 < this.getEffectiveSizeC(); ++c2) {
                            int index = this.getIndex(z, c2, t);
                            store.setPlaneDeltaT(deltaT, 0, index);
                        }
                    }
                }
            }
            for (int i = 0; i < this.getEffectiveSizeC(); ++i) {
                Length ex;
                Length em;
                if (channelNames.containsKey(i)) {
                    store.setChannelName((String)channelNames.get(i), 0, i);
                }
                if (pinholes.containsKey(i)) {
                    store.setChannelPinholeSize(new Length((Number)pinholes.get(i), UNITS.MICROMETER), 0, i);
                }
                if (emWaves != null && i < emWaves.length && (em = FormatTools.getEmissionWavelength(emWaves[i])) != null) {
                    store.setChannelEmissionWavelength(em, 0, i);
                }
                if (exWaves == null || i >= exWaves.length || (ex = FormatTools.getExcitationWavelength(exWaves[i])) == null) continue;
                store.setChannelExcitationWavelength(ex, 0, i);
            }
            Object[] lasers = wavelengths.keySet().toArray(new Integer[0]);
            Arrays.sort(lasers);
            for (int i = 0; i < lasers.length; ++i) {
                Frequency theLaserRepetitionRate;
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, i), 0, i);
                Length wave = FormatTools.getWavelength((Double)wavelengths.get(lasers[i]));
                if (wave != null) {
                    store.setLaserWavelength(wave, 0, i);
                }
                store.setLaserType(this.getLaserType("Other"), 0, i);
                store.setLaserLaserMedium(this.getLaserMedium("Other"), 0, i);
                store.setLaserManufacturer((String)laserManufacturer, 0, i);
                store.setLaserModel((String)laserModel, 0, i);
                Power theLaserPower = FormatTools.createPower(laserPower, UNITS.MILLIWATT);
                if (theLaserPower != null) {
                    store.setLaserPower(theLaserPower, 0, i);
                }
                if ((theLaserRepetitionRate = FormatTools.createFrequency(laserRepetitionRate, UNITS.HERTZ)) == null) continue;
                store.setLaserRepetitionRate(theLaserRepetitionRate, 0, i);
            }
            if (lasers.length == 0 && laserManufacturer != null) {
                Frequency theLaserRepetitionRate;
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, 0), 0, 0);
                store.setLaserType(this.getLaserType("Other"), 0, 0);
                store.setLaserLaserMedium(this.getLaserMedium("Other"), 0, 0);
                store.setLaserManufacturer((String)laserManufacturer, 0, 0);
                store.setLaserModel((String)laserModel, 0, 0);
                Power theLaserPower = FormatTools.createPower(laserPower, UNITS.MILLIWATT);
                if (theLaserPower != null) {
                    store.setLaserPower(theLaserPower, 0, 0);
                }
                if ((theLaserRepetitionRate = FormatTools.createFrequency(laserRepetitionRate, UNITS.HERTZ)) != null) {
                    store.setLaserRepetitionRate(theLaserRepetitionRate, 0, 0);
                }
            }
            if (filterSetModel != null) {
                store.setFilterSetID(MetadataTools.createLSID("FilterSet", 0, 0), 0, 0);
                store.setFilterSetModel((String)filterSetModel, 0, 0);
                String dichroicID = MetadataTools.createLSID("Dichroic", 0, 0);
                String emFilterID = MetadataTools.createLSID("Filter", 0, 0);
                String exFilterID = MetadataTools.createLSID("Filter", 0, 1);
                store.setDichroicID(dichroicID, 0, 0);
                store.setDichroicModel((String)dichroicModel, 0, 0);
                store.setFilterSetDichroicRef(dichroicID, 0, 0);
                store.setFilterID(emFilterID, 0, 0);
                store.setFilterModel((String)emissionModel, 0, 0);
                store.setFilterSetEmissionFilterRef(emFilterID, 0, 0, 0);
                store.setFilterID(exFilterID, 0, 1);
                store.setFilterModel((String)excitationModel, 0, 1);
                store.setFilterSetExcitationFilterRef(exFilterID, 0, 0, 0);
            }
            if (objectiveModel != null) {
                store.setObjectiveModel((String)objectiveModel, 0, 0);
            }
            if (immersion == null) {
                immersion = "Other";
            }
            store.setObjectiveImmersion(this.getImmersion((String)immersion), 0, 0);
            if (lensNA != null) {
                store.setObjectiveLensNA(lensNA, 0, 0);
            }
            if (workingDistance != null) {
                store.setObjectiveWorkingDistance(new Length(workingDistance, UNITS.MICROMETER), 0, 0);
            }
            if (magnification != null) {
                store.setObjectiveCalibratedMagnification(magnification, 0, 0);
            }
            store.setObjectiveCorrection(this.getCorrection("Other"), 0, 0);
            String objectiveID = MetadataTools.createLSID("Objective", 0, 0);
            store.setObjectiveID(objectiveID, 0, 0);
            store.setObjectiveSettingsID(objectiveID, 0);
            String detectorID = MetadataTools.createLSID("Detector", 0, 0);
            store.setDetectorID(detectorID, 0, 0);
            store.setDetectorManufacturer((String)detectorManufacturer, 0, 0);
            store.setDetectorModel((String)detectorModel, 0, 0);
            store.setDetectorType(this.getDetectorType("Other"), 0, 0);
            for (Integer key : gains.keySet()) {
                int index = key;
                if (index >= this.getEffectiveSizeC()) continue;
                store.setDetectorSettingsGain((Double)gains.get(key), 0, index);
                store.setDetectorSettingsID(detectorID, 0, index);
            }
            if (lastName != null) {
                String experimenterID = MetadataTools.createLSID("Experimenter", 0);
                store.setExperimenterID(experimenterID, 0);
                store.setExperimenterLastName((String)lastName, 0);
            }
            if (stagePos != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    if (stagePos.length > 0) {
                        store.setPlanePositionX(stagePos[0], 0, i);
                        this.addGlobalMeta("X position for position #1", stagePos[0]);
                    }
                    if (stagePos.length > 1) {
                        store.setPlanePositionY(stagePos[1], 0, i);
                        this.addGlobalMeta("Y position for position #1", stagePos[1]);
                    }
                    if (stagePos.length <= 2) continue;
                    store.setPlanePositionZ(stagePos[2], 0, i);
                    this.addGlobalMeta("Z position for position #1", stagePos[2]);
                }
            }
            if (exposureTime != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    store.setPlaneExposureTime(exposureTime, 0, i);
                }
            }
        }
    }

    private String[] tokenize(String line) {
        ArrayList<String> tokens = new ArrayList<String>();
        boolean inWhiteSpace = true;
        boolean withinQuotes = false;
        StringBuilder token = null;
        for (int i = 0; i < line.length(); ++i) {
            char c = line.charAt(i);
            if (Character.isWhitespace(c) || c == '\u0004') {
                if (withinQuotes) {
                    token.append(c);
                    continue;
                }
                if (inWhiteSpace) continue;
                inWhiteSpace = true;
                if (token.length() <= 0) continue;
                tokens.add(token.toString());
                token = null;
                continue;
            }
            if ('\"' == c) {
                boolean bl = withinQuotes = !withinQuotes;
            }
            if (inWhiteSpace) {
                inWhiteSpace = false;
                token = new StringBuilder();
            }
            token.append(c);
        }
        if (null != token && token.length() > 0) {
            tokens.add(token.toString());
        }
        return tokens.toArray(new String[0]);
    }

    String[] findKeyValue(String[] tokens, String[][] regexesArray) {
        String[] keyValue = this.findKeyValueForCategory(tokens, regexesArray);
        if (null == keyValue) {
            keyValue = this.findKeyValueOther(tokens, OTHER_KEYS);
        }
        if (null == keyValue) {
            String key = tokens[0];
            String value = this.concatenateTokens(tokens, 1, tokens.length);
            keyValue = new String[]{key, value};
        }
        return keyValue;
    }

    private String concatenateTokens(String[] tokens, int start, int stop) {
        StringBuilder returnValue = new StringBuilder();
        for (int i = start; i < tokens.length && i < stop; ++i) {
            returnValue.append(tokens[i]);
            if (i >= stop - 1) continue;
            returnValue.append(' ');
        }
        return returnValue.toString();
    }

    private String[] findKeyValueForCategory(String[] tokens, String[][] regexesArray) {
        String[] keyValue = null;
        int index = 0;
        for (String[] regexes : regexesArray) {
            if (this.compareTokens(tokens, 1, regexes, 0)) {
                int splitIndex = 1 + regexes.length;
                String key = this.concatenateTokens(tokens, 0, splitIndex);
                String value = this.concatenateTokens(tokens, splitIndex, tokens.length);
                keyValue = new String[]{key, value};
                break;
            }
            ++index;
        }
        return keyValue;
    }

    private String[] findKeyValueOther(String[] tokens, String[][] regexesArray) {
        String[] keyValue = null;
        for (String[] regexes : regexesArray) {
            for (int i = 1; i < tokens.length - regexes.length; ++i) {
                if (!tokens[i].toLowerCase().matches(regexes[0]) || 1 != regexes.length && !this.compareTokens(tokens, i + 1, regexes, 1)) continue;
                int splitIndex = i + regexes.length;
                String key = this.concatenateTokens(tokens, 0, splitIndex);
                String value = this.concatenateTokens(tokens, splitIndex, tokens.length);
                keyValue = new String[]{key, value};
                break;
            }
            if (null != keyValue) break;
        }
        return keyValue;
    }

    private boolean compareTokens(String[] tokens, int tokenIndex, String[] regexes, int regexesIndex) {
        boolean returnValue = true;
        int i = tokenIndex;
        for (int j = regexesIndex; j < regexes.length; ++j) {
            if (i >= tokens.length || !tokens[i].toLowerCase().matches(regexes[j])) {
                returnValue = false;
                break;
            }
            ++i;
        }
        return returnValue;
    }

    private Double[] splitDoubles(String v) {
        StringTokenizer t = new StringTokenizer(v);
        Double[] values = new Double[t.countTokens()];
        for (int n = 0; n < values.length; ++n) {
            String token = t.nextToken().trim();
            try {
                values[n] = new Double(token);
                continue;
            }
            catch (NumberFormatException e) {
                LOGGER.debug("Could not parse double value '{}'", (Object)token, (Object)e);
            }
        }
        return values;
    }

    private boolean checkUnit(String actual, String ... expected) {
        if (actual == null || actual.equals("")) {
            return true;
        }
        for (String exp : expected) {
            if (!actual.equals(exp)) continue;
            return true;
        }
        LOGGER.debug("Unexpected unit '{}'; expected '{}'", (Object)actual, (Object)expected);
        return false;
    }
}

